// $Id: HNSCocoaScript.h,v 1.12 2003/03/20 13:39:57 hns Exp $
//
//  CocoaScript.h
//  CocoaScript
//
//  Created by Dr. H. Nikolaus Schaller on Sat Jun 15 2002.
//  Copyright (c) 2001 __MyCompanyName__. All rights reserved.
//

//
// a CocoaScript instance is part of a syntax tree that can be evaluated
//
// basically it represents [left method:right] of Objective-C
//
// evaluation works as follows
// 1. the left is sent the eval method - if method exists; otherwise the object is used untouched
// 2. then, all right arguments are sent the eval method, i.e. executing them as well
// 3. the method is called (if not nil) for the evaluated left object and passing all right objects
// 4. the result is returned
//

#import <Cocoa/Cocoa.h>

@interface HNSCocoaScript : NSObject <NSCoding> {
	id left;						// object to send message to (nil for a script)
	SEL method;						// method to call (nil for a script)
	NSMutableArray *right;			// list of arguments or script sequence objects
//	NSMutableDictionary *vars;		// variables
}

// get components
- (SEL) method;
- (id) left;
- (NSArray *) right;

// runtime interpreter
- (id) eval;									// evaluate without arguments
- (id) evaluate:(NSArray *) args;				// evaluate with arguments (args[0] is self, args[1] is _cmd)

- (void) CSloop;									// run script in infinite loop with restarting from beginning
- (void) CSexception:(HNSCocoaScript *) handler;	// try script and call handler on exceptions
- (id) CSfunction;									// function wrapper block; returns value of "retval"
- (id) CSEVALSHIELD;								// returns script object instead of result

// local variables
- (id) valueForName:(NSString *) name;						// get variable value for that name
- (void) setValueForName:(NSString *) name to:(id) object;	// set variable for that name

// debugger mode
- (void) invoke:(NSArray *) args;				// invoke with arguments but don't execute
- (BOOL) singleStep;							// evaluate single step - return YES if more statements follow

+ (NSArray *) stack;							// return shared program flow stack (for debugging purposes)
+ (void) clearStack;							// clear shared stack

// compiler
+ (HNSCocoaScript *) scriptFromScanner:(NSScanner *) s;	// compile and return generated object script
+ (HNSCocoaScript *) scriptForString:(NSString *) s;	// get script for string
- (void) clearScript;							// clear script, i.e. make empty skript - but keep variables!
- (NSArray *) code;								// get compiled code
- (NSScanner *) scanner;						// access (shared!) scanner to determine error position

// attach script to method in class hierarchy
+ (void) defineMethod:(NSString *) method ofType:(Class) type andArgs:(NSArray *) arg asScript:(HNSCocoaScript *) code;	// make script called when (class/instance) method is invoked

// convert to Objective C
- (NSString *) makeObjectiveC;

@end

@interface NSObject (HNSCocoaScript)
- (BOOL) boolValue;					// get boolean value
- (int) intValue;					// get integer value
- (long) longValue;					// get long integer value
- (double) doubleValue;				// get double value
- (NSString *) stringValue;			// get value as string
- (void) msgbox;					// show description in a message box
- (id) CSIF:(HNSCocoaScript *) t;
- (id) CSIF:(HNSCocoaScript *) t ELSE:(HNSCocoaScript *) f;
- (id) CSSELECT:(NSDictionary *) t;
- (id) CSSELECT:(NSDictionary *) t DEFAULT:(HNSCocoaScript *) f;
@end

@interface NSNumber (HNSCocoaScript)
- (id) CSadd:(id) b;	// b must be number as well
- (id) CSsub:(id) b;	// b must be number as well
- (id) CSmul:(id) b;	// b must be number as well
- (id) CSdiv:(id) b;	// b must be number as well
- (id) CSand:(id) b;	// b must be boolean as well
- (id) CSor:(id) b;		// b must be boolean as well
- (id) CSxor:(id) b;	// b must be boolean as well
- (id) CSchs;			// change sign
- (id) CSnot;			// boolean not
- (id) CSle:(id) b;		// b must be number as well
- (id) CSlt:(id) b;		// b must be number as well
- (id) CSge:(id) b;		// b must be number as well
- (id) CSgt:(id) b;		// b must be number as well
- (id) CSeq:(id) b;		// b must be number as well
- (id) CSne:(id) b;		// b must be number as well
- (NSString *) str;
@end

@interface NSString (HNSCocoaScript)
- (id) CSadd:(id) b;		// b must be string as well
- (id) CSeq:(id) b;			// b must be string as well
- (id) CSne:(id) b;			// b must be string as well
- (NSString *) encodeAsHTML;					// encode special characters and Unicode as &#d; and \n as <br>
- (NSString *) encodeAsURL;						// escape blank, ?, = etc. as %h
- (void) showurl;			// interpret string as URL and show
- (id) len;					// length
- (id) left:(id) count;		// left part of string
- (id) right:(id) count;	// right part of string
- (id) mid:(id) pos;		// middle part of string
- (id) mid:(id) pos length:(id) count;
- (Class) findClass;		// "string".findClass
@end

@interface NSMutableString (HNSCocoaScript)
- (void) setMid:(id) pos length:(id) count to:(NSString *) value;	// left side mid - replace range of characters
@end

@interface NSArray (HNSCocoaScript)
- (id) CSadd:(id) b;	// b must be array as well
- (unsigned) origin;	// array index origin - returns 0
@end

@interface NSDictionary (HNSCocoaScript)
- (id) CSadd:(id) b;	// b must be dictionary as well
@end

@interface NSInvocation (HNSCocoaBasic)
- (void)setVariableArgument:(void *) argumentLocation ofType:(enum _NSObjCValueType) argumentType;
@end

